<?php

namespace App\Livewire\Business;

use App\Classes\Country;
use App\Models\PaymentGateway;
use App\Models\Tax;
use App\Traits\LivewireToastr;
use Livewire\Component;

class Checkout extends Component
{
    use LivewireToastr;

    public $businessOwner;
    public $trx;
    public $summary = [];
    public $payment_method;

    public $address_line_1;
    public $address_line_2;
    public $city;
    public $state;
    public $zip;
    public $country;

    public function mount()
    {
        $this->businessOwner = authBusinessOwner();
        $this->payment_method = old('payment_method') ?? PaymentGateway::active()->first()->alias;

        $businessOwner = $this->businessOwner;
        $this->address_line_1 = old('address_line_1') ?? @$businessOwner->address->line_1;
        $this->address_line_2 = old('address_line_2') ?? @$businessOwner->address->line_2;
        $this->city = old('city') ?? @$businessOwner->address->city;
        $this->state = old('state') ?? @$businessOwner->address->state;
        $this->zip = old('zip') ?? @$businessOwner->address->zip;
        $this->country = old('country') ?? @$businessOwner->address->country;
    }

    public function updateSummary()
    {
        $trx = $this->trx;
        $total = $trx->amount;

        $tax = null;
        if ($this->country) {
            $tax = Tax::whereJsonContains('countries', $this->country)->first();

            if ($tax) {
                $taxRate = $tax->rate;
                $taxAmount = ($total * $taxRate) / 100;

                $tax = [
                    'name' => $tax->trans->name,
                    'rate' => $taxRate,
                    'amount' => getAmount(round($taxAmount, 2)),
                ];

                $total = round($total + $taxAmount, 2);
            }
        }

        $gateway = null;
        if ($this->payment_method) {

            $paymentGateway = PaymentGateway::where('alias', $this->payment_method)
                ->active()->first();

            if ($paymentGateway) {
                $gatewayFees = $paymentGateway->fees;

                if ($gatewayFees > 0) {
                    $feesAmount = ($total * $gatewayFees) / 100;

                    $gateway = [
                        'name' => $paymentGateway->trans->name,
                        'fees' => $gatewayFees,
                        'amount' => getAmount(round($feesAmount, 2)),
                    ];

                    $total = round($total + $feesAmount, 2);
                }
            }
        }

        $this->summary = [
            'tax' => $tax,
            'gateway' => $gateway,
            'total' => $total,
        ];
    }

    public function render()
    {
        $this->updateSummary();

        $paymentGateways = PaymentGateway::active()->get();

        return theme_view('livewire.business.checkout', [
            'paymentGateways' => $paymentGateways,
        ]);
    }
}